import React, { Component } from 'react';
import { History } from 'history';
import 'react-circular-progressbar/dist/styles.css';
import { getCode, getNames, getName } from 'country-list';
import { withStyles } from '@material-ui/core/styles';
import { Logger } from 'library/Logger';
import routes from 'constants/routes';
import { shell } from 'electron';
import logo from 'resources/images/logo_with_text.png';
import checkBoxOn from 'resources/images/check_box.svg';
import checkBoxOff from 'resources/images/check_box_outline.svg';
import Select from '@material-ui/core/Select';
import MenuItem from '@material-ui/core/MenuItem';
import cn from 'classnames';
import { CircularProgress, TextField } from '@material-ui/core';
import Button from '@material-ui/core/Button';
import fillTemplate from 'es6-dynamic-template';
import Interweave from 'interweave';
import settings from 'electron-settings';
import { Autocomplete } from '@material-ui/lab';
import { DateTime } from 'luxon';
import {
    ALL_LANGUAGES,
    DEFAULT_LANGUAGE_CODE
} from '../../resources/strings/languages';
import { MODELS } from '../../library/headset/consts';
import {
    PRIVACY_URL,
    SUPPORT_URL,
    TERRANO_SUPPORT_URL,
    TOS_URL
} from '../../constants/urls';
import * as Authentication from '../../library/Authentication';
import backIcon from '../../resources/images/arrow_back.png';
import supportIcon from '../../resources/images/help_filled.svg';
import styles from './styles';
import { getPlatformOs } from '../../library/Authentication';

type Props = {
    device: object,
    language: object,
    strings: object,
    history: History,
    onAuthenticationChange: () => {},
    onLanguageSet: () => {},
    classes: {
        [key: string]: string
    }
};

// How long to wait for an email verification (mins)
const MAX_WAIT_TIME_FOR_VERIFICATION_MINS = 30;

class WelcomeScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            device: props.device,
            language: props.language,
            strings: props.strings,
            waitingForVerification: false,
            showSignIn: false,
            showSignUpContinued: false,
            loading: false,
            emailErrorText: null,
            firstNameErrorText: null,
            lastNameErrorText: null,
            generalErrorText: null,
            verifyEmail: false,
            resendingEmail: false,
            checkForVerificationInterval: null,
            email: '',
            firstName: '',
            lastName: '',
            countryCode: null,
            marketingApproval: false,
            furtherRegistrationNeeded: false
        };
    }

    componentDidMount() {
        const verificationEmail = settings.getSync('verification_email');
        const verificationTimestamp = settings.getSync('verification_time');
        const backToSignIn = settings.getSync('verification_return_to_sign_in');

        if (verificationEmail && verificationTimestamp) {
            const lastVerification = DateTime.fromMillis(verificationTimestamp);
            const now = DateTime.now();
            const timePassed = now.diff(lastVerification, ['minutes']).minutes;

            if (timePassed <= MAX_WAIT_TIME_FOR_VERIFICATION_MINS) {
                Logger.info(
                    'User previously tried to verify email but closed app'
                );
                this.setState({
                    email: verificationEmail,
                    verifyEmail: true,
                    showSignIn: backToSignIn,
                    checkForVerificationInterval: setInterval(() => {
                        this.checkForVerification();
                    }, 1000)
                });
            }
        }
    }

    componentDidUpdate(prevProps, prevState) {
        const { device, strings, language } = this.props;
        Logger.info(
            'componentDidUpdate - prevDevice',
            prevState.device,
            'newDevice',
            device
        );

        if (prevState.language !== language) {
            this.setState({ language, strings });
        }
    }

    componentWillUnmount() {
        const { checkForVerificationInterval } = this.state;
        if (checkForVerificationInterval) {
            clearInterval(checkForVerificationInterval);
        }
    }

    moveToScreen(path) {
        const { history } = this.props;
        Logger.info('NotLoggedInScreen: moveToScreen: ', path);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(path);
        }, 100);
    }

    renderLanguageController() {
        const { language, onLanguageSet, classes } = this.props;

        return (
            <Select
                className={cn(classes.language, classes.languageSelect)}
                value={language}
                onChange={event => onLanguageSet(event.target.value)}
                MenuProps={{
                    classes: {
                        paper: classes.languagePaper
                    }
                }}
                disableUnderline
                inputProps={{
                    classes: {
                        icon: classes.languageInputArrow,
                        root: classes.languageInput,
                        select: classes.languageSelect
                    }
                }}
            >
                {Object.keys(ALL_LANGUAGES).map(key => (
                    <MenuItem
                        className={cn(
                            classes.languageMenuItem,
                            key === language ? classes.selectedMenuItem : ''
                        )}
                        key={key}
                        value={key}
                    >
                        <img alt={key} src={ALL_LANGUAGES[key].icon} />{' '}
                        {ALL_LANGUAGES[key].title}
                    </MenuItem>
                ))}
            </Select>
        );
    }

    validateEmail(email) {
        const re = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(String(email).toLowerCase());
    }

    checkForVerification() {
        const {
            furtherRegistrationNeeded,
            checkForVerificationInterval
        } = this.state;
        const self = this;

        async function func() {
            try {
                const [
                    signedIn,
                    response
                ] = await Authentication.checkIfSignedIn();
                Logger.info(`Signed In: ${signedIn}: ${response}`);
                if (signedIn) {
                    Logger.info(`Signed In: ${self} - ${self.moveToScreen}`);

                    if (response.registration_required) {
                        Logger.info('Further registration details missing');
                        clearInterval(checkForVerificationInterval);

                        self.setState({
                            firstName: response.first_name || '',
                            lastName: response.last_name || '',
                            language:
                                response.language || DEFAULT_LANGUAGE_CODE,
                            countryCode: response.country || null,
                            marketingApproval:
                                response.marketing_approval !== null &&
                                response.marketing_approval !== undefined
                                    ? response.marketing_approval
                                    : false,
                            furtherRegistrationNeeded: true,
                            checkForVerificationInterval: null
                        });
                    } else {
                        settings.unsetSync('verification_email');
                        settings.unsetSync('verification_time');
                        settings.unsetSync('verification_return_to_sign_in');

                        self.props.onAuthenticationChange();
                        self.moveToScreen(routes.WELCOME.path);
                    }
                }
            } catch (e) {
                Logger.error(`Error checking if signed in: ${e}`);
            }
        }

        if (!furtherRegistrationNeeded) {
            func();
        }
    }

    renderSignIn() {
        const { classes, strings } = this.props;
        const {
            email,
            loading,
            language,
            emailErrorText,
            showSignIn
        } = this.state;
        const emailValidated =
            email && email.length > 0 && this.validateEmail(email);

        const signIn = async () => {
            if (loading) return;
            if (!emailValidated) return;

            this.setState({ loading: true });
            try {
                await Authentication.signIn(email, language);
                settings.setSync('verification_email', email);
                settings.setSync(
                    'verification_time',
                    DateTime.now().toMillis()
                );
                settings.setSync('verification_return_to_sign_in', showSignIn);
                this.setState({
                    verifyEmail: true,
                    checkForVerificationInterval: setInterval(() => {
                        this.checkForVerification();
                    }, 1000)
                });
            } catch (e) {
                if (e === Authentication.API_ERROR_USER_NOT_FOUND) {
                    this.setState({
                        emailErrorText: strings.email_doesnt_exist
                    });
                } else {
                    this.setState({
                        emailErrorText: strings.network_error
                    });
                }
            }

            this.setState({ loading: false });
        };

        return (
            <div className={classes.signInContainer}>
                <div className={classes.signInTitle}>{strings.sign_in}</div>
                <TextField
                    className={classes.signInEmail}
                    placeholder={strings.email}
                    value={email}
                    onKeyDown={e => {
                        if (e.keyCode === 13) signIn();
                    }}
                    onKeyPress={e => {
                        // Prevent spaces from being entered
                        if (e.which === 32) {
                            e.preventDefault();
                            return false;
                        }
                        return true;
                    }}
                    onChange={e =>
                        this.setState({
                            email: e.target.value
                        })
                    }
                    onBlur={() => {
                        this.setState({
                            emailErrorText: this.validateEmail(email)
                                ? null
                                : strings.invalid_email_format
                        });
                    }}
                    error={emailErrorText}
                    helperText={emailErrorText}
                />
                <Button
                    className={classes.signInButton}
                    variant="contained"
                    color="primary"
                    disabled={!emailValidated}
                    onClick={signIn}
                >
                    {loading ? (
                        <CircularProgress
                            className={classes.loadingButton}
                            size={20}
                        />
                    ) : (
                        strings.sign_in
                    )}
                </Button>
                <div className={classes.signUpContainer}>
                    {strings.dont_have_an_account}
                    <div
                        className={classes.signUpLink}
                        onClick={() =>
                            this.setState({
                                showSignIn: false,
                                emailErrorText: null
                            })
                        }
                    >
                        {strings.sign_up}
                    </div>
                </div>
            </div>
        );
    }

    renderVerifyEmail() {
        const { classes, strings } = this.props;
        const { checkForVerificationInterval, showSignIn } = this.state;

        return (
            <div className={classes.signInContainer}>
                <div
                    className={classes.backContainer}
                    onClick={() => {
                        clearInterval(checkForVerificationInterval);

                        settings.unsetSync('verification_email');
                        settings.unsetSync('verification_time');
                        settings.unsetSync('verification_return_to_sign_in');

                        const newState = {
                            verifyEmail: false,
                            loading: false,
                            checkForVerificationInterval: null
                        };

                        if (showSignIn) {
                            // Back to login
                            newState.showSignIn = true;
                        } else {
                            // Back to sign up (step 1)
                            newState.showSignIn = false;
                            newState.showSignUpContinued = false;
                            newState.email = '';
                            newState.firstName = '';
                            newState.lastName = '';
                            newState.countryCode = null;
                            newState.marketingApproval = false;
                        }

                        this.setState(newState);
                    }}
                >
                    <img
                        className={classes.backIcon}
                        src={backIcon}
                        alt="Back"
                    />
                    {showSignIn
                        ? strings.back_to_sign_in
                        : strings.back_to_sign_up}
                </div>

                <div className={classes.verifyContainer}>
                    <div className={classes.signInTitle}>
                        {strings.verify_your_email_address}
                    </div>
                    <div className={classes.verifySubTitle}>
                        <Interweave
                            content={fillTemplate(
                                strings.we_sent_verification_email,
                                { email: Authentication.getUserEmail() }
                            )}
                        />
                    </div>
                    <div className={classes.verifyWarning}>
                        {strings.do_not_close_app}
                    </div>
                </div>

                <div className={classes.signUpContainer}>
                    {strings.didnt_receive_an_email}
                    <div
                        className={classes.signUpLink}
                        onClick={async () => {
                            const { language } = this.state;
                            const email = Authentication.getUserEmail();

                            this.setState({ resendingEmail: true });
                            try {
                                await Authentication.signIn(email, language);
                            } catch (e) {
                                Logger.error(e);
                            }
                            this.setState({ resendingEmail: false });
                        }}
                    >
                        {strings.send_again}
                    </div>
                </div>
            </div>
        );
    }

    renderResendingEmail() {
        const { classes } = this.props;

        return (
            <div className={classes.resendingEmailContainer}>
                <CircularProgress size={80} />
            </div>
        );
    }

    validateName(name) {
        const re = /^\p{L}+[\p{L} ]*$/u;
        return re.test(String(name)) && name.length <= 50;
    }

    renderSignUp() {
        const { classes, strings, onAuthenticationChange } = this.props;
        const {
            email,
            firstName,
            lastName,
            loading,
            emailErrorText,
            firstNameErrorText,
            lastNameErrorText,
            showSignUpContinued,
            countryCode,
            marketingApproval,
            language,
            generalErrorText,
            furtherRegistrationNeeded,
            showSignIn
        } = this.state;
        const formValidated =
            email.length > 0 &&
            this.validateEmail(email) &&
            firstName.length > 0 &&
            this.validateName(firstName) &&
            lastName.length > 0 &&
            this.validateName(lastName);
        const formContinuedValidated =
            countryCode && countryCode.length > 0 && countryCode !== null;

        const signUp = async () => {
            if (loading) return;
            if (!formContinuedValidated) return;

            this.setState({ loading: true, generalErrorText: null });

            try {
                if (furtherRegistrationNeeded) {
                    // Just set the missing user details
                    await Authentication.setUserInfo({
                        first_name: firstName,
                        last_name: lastName,
                        country: countryCode,
                        language,
                        marketing_approval: marketingApproval,
                        platform_os: getPlatformOs()
                    });

                    settings.unsetSync('verification_email');
                    settings.unsetSync('verification_time');
                    settings.unsetSync('verification_return_to_sign_in');

                    onAuthenticationChange();
                    this.moveToScreen(routes.WELCOME.path);
                } else {
                    // Full sign up
                    await Authentication.signUp(
                        email,
                        firstName,
                        lastName,
                        countryCode,
                        language,
                        marketingApproval
                    );
                    settings.setSync('verification_email', email);
                    settings.setSync(
                        'verification_time',
                        DateTime.now().toMillis()
                    );
                    settings.setSync(
                        'verification_return_to_sign_in',
                        showSignIn
                    );
                    this.setState({
                        verifyEmail: true,
                        checkForVerificationInterval: setInterval(() => {
                            this.checkForVerification();
                        }, 1000)
                    });
                }
            } catch (e) {
                this.setState({
                    generalErrorText: strings.network_error
                });
            }

            this.setState({ loading: true });
        };

        const checkEmailNotRegistered = async () => {
            if (loading) return;
            if (!formValidated) return;

            if (furtherRegistrationNeeded) {
                // No need to check if email is already registered (we're at the complete-your-details phase for
                // an already registered user)
                this.setState({ showSignUpContinued: true });
                return;
            }

            this.setState({ loading: true });

            try {
                const exists = await Authentication.checkIfEmailExists(email);
                if (exists) {
                    // Email already exists
                    this.setState({
                        emailErrorText: strings.email_already_registered
                    });
                } else {
                    this.setState({ showSignUpContinued: true });
                }
            } catch (e) {
                this.setState({
                    emailErrorText: strings.network_error
                });
            }

            this.setState({ loading: false });
        };

        let buttonText = showSignUpContinued
            ? strings.sign_up
            : strings.continue;
        if (furtherRegistrationNeeded && showSignUpContinued) {
            buttonText = strings.done_button;
        }

        const getCountryIcon = countryName => {
            const cleanName = countryName
                .toLowerCase()
                .replace(/ /g, '-')
                .replace(/'/g, '');
            // eslint-disable-next-line global-require,import/no-dynamic-require
            return require(`resources/images/countries/${cleanName}.svg`);
        };

        return (
            <div className={classes.signInContainer}>
                <div className={classes.signInTitle}>
                    {furtherRegistrationNeeded
                        ? strings.complete_your_details
                        : strings.sign_up}
                </div>
                {!showSignUpContinued && (
                    <>
                        <TextField
                            className={classes.signInEmail}
                            placeholder={strings.email}
                            value={email}
                            disabled={furtherRegistrationNeeded}
                            onChange={e =>
                                this.setState({
                                    email: e.target.value
                                })
                            }
                            onKeyPress={e => {
                                // Prevent spaces from being entered
                                if (e.which === 32) {
                                    e.preventDefault();
                                    return false;
                                }
                                return true;
                            }}
                            onBlur={() => {
                                this.setState({
                                    emailErrorText: this.validateEmail(email)
                                        ? null
                                        : strings.invalid_email_format
                                });
                            }}
                            error={emailErrorText}
                            helperText={emailErrorText}
                        />
                        <TextField
                            className={classes.signInEmail}
                            placeholder={strings.first_name}
                            value={firstName}
                            inputProps={{ maxLength: 50 }}
                            onChange={e =>
                                this.setState({
                                    firstName: e.target.value,
                                    firstNameErrorText: this.validateName(
                                        e.target.value
                                    )
                                        ? null
                                        : strings.letters_only
                                })
                            }
                            error={firstNameErrorText}
                            helperText={firstNameErrorText}
                        />
                        <TextField
                            className={classes.signInEmail}
                            placeholder={strings.last_name}
                            value={lastName}
                            inputProps={{ maxLength: 50 }}
                            onKeyDown={e => {
                                if (e.keyCode === 13) checkEmailNotRegistered();
                            }}
                            onChange={e =>
                                this.setState({
                                    lastName: e.target.value,
                                    lastNameErrorText: this.validateName(
                                        e.target.value
                                    )
                                        ? null
                                        : strings.letters_only
                                })
                            }
                            error={lastNameErrorText}
                            helperText={lastNameErrorText}
                        />
                    </>
                )}
                {showSignUpContinued && (
                    <>
                        <div
                            className={classes.backContainer}
                            onClick={() =>
                                this.setState({
                                    showSignUpContinued: false
                                })
                            }
                        >
                            <img
                                alt="Back"
                                className={classes.backIcon}
                                src={backIcon}
                            />
                            {strings.back}
                        </div>

                        <Autocomplete
                            options={getNames().sort()}
                            classes={{
                                option: classes.option
                            }}
                            onChange={(_, v) => {
                                this.setState({
                                    countryCode: v ? getCode(v) : null
                                });
                            }}
                            noOptionsText={strings.no_results}
                            autoHighlight
                            getOptionLabel={option => option}
                            value={countryCode ? getName(countryCode) : null}
                            renderOption={option => (
                                <React.Fragment>
                                    <img
                                        className={classes.countryIcon}
                                        alt={option}
                                        src={getCountryIcon(option)}
                                    />
                                    {option}
                                </React.Fragment>
                            )}
                            renderInput={params => {
                                const newParams = params;
                                delete newParams.InputProps.endAdornment;
                                return (
                                    <TextField
                                        {...newParams}
                                        placeholder={strings.country}
                                        className={classes.signInEmail}
                                        inputProps={{
                                            ...params.inputProps,
                                            autoComplete: 'new-password' // disable autocomplete and autofill
                                        }}
                                    />
                                );
                            }}
                        />

                        <div
                            className={classes.marketingApprovalContainer}
                            onClick={() => {
                                this.setState({
                                    marketingApproval: !marketingApproval
                                });
                            }}
                        >
                            <img
                                alt="Checkbox"
                                className={classes.marketingApprovalIcon}
                                src={
                                    marketingApproval ? checkBoxOn : checkBoxOff
                                }
                            />
                            <div className={classes.marketingApprovalText}>
                                {strings.marketing_approval}
                            </div>
                        </div>
                    </>
                )}
                <Button
                    className={classes.signInButton}
                    variant="contained"
                    color="primary"
                    disabled={
                        showSignUpContinued
                            ? !formContinuedValidated
                            : !formValidated
                    }
                    onClick={
                        showSignUpContinued ? signUp : checkEmailNotRegistered
                    }
                >
                    {loading ? (
                        <CircularProgress
                            className={classes.loadingButton}
                            size={20}
                        />
                    ) : (
                        buttonText
                    )}
                </Button>
                {generalErrorText && (
                    <div className={classes.generalError}>
                        {generalErrorText}
                    </div>
                )}
                {showSignUpContinued && !furtherRegistrationNeeded && (
                    <div className={classes.agreeTOS}>
                        <Interweave
                            content={fillTemplate(strings.by_clicking_sign_up, {
                                privacy_url: PRIVACY_URL,
                                tos_url: TOS_URL
                            })}
                        />
                    </div>
                )}
                {!furtherRegistrationNeeded && (
                    <div className={classes.signUpContainer}>
                        {strings.already_have_an_account}
                        <div
                            className={classes.signUpLink}
                            onClick={() =>
                                this.setState({
                                    showSignIn: true,
                                    showSignUpContinued: false,
                                    emailErrorText: null
                                })
                            }
                        >
                            {strings.sign_in}
                        </div>
                    </div>
                )}
            </div>
        );
    }

    renderRightSide() {
        const {
            showSignIn,
            verifyEmail,
            resendingEmail,
            furtherRegistrationNeeded
        } = this.state;

        return (
            <>
                {resendingEmail && this.renderResendingEmail()}
                {(!verifyEmail || furtherRegistrationNeeded) &&
                    this.renderLanguageController()}
                {showSignIn && !verifyEmail && this.renderSignIn()}
                {!showSignIn && !verifyEmail && this.renderSignUp()}
                {furtherRegistrationNeeded && this.renderSignUp()}
                {verifyEmail &&
                    !furtherRegistrationNeeded &&
                    this.renderVerifyEmail()}
            </>
        );
    }

    render() {
        const {
            strings,
            device,
            waitingForVerification,
            furtherRegistrationNeeded
        } = this.state;
        const { classes } = this.props;

        Logger.info('NotLoggedInScreen: render');

        if (Authentication.isSignedIn() && !furtherRegistrationNeeded) {
            Logger.info('NotLoggedInScreen: is logged in');
            if (!Authentication.isWaitingForEmailVerification()) {
                settings.unsetSync('verification_email');
                settings.unsetSync('verification_time');
                settings.unsetSync('verification_return_to_sign_in');

                this.moveToScreen(routes.WELCOME.path);
            } else if (!waitingForVerification) {
                this.setState({ waitingForVerification: true });
            }
        }

        return (
            <div className={classes.container}>
                <div className={classes.leftContainer}>
                    <img className={classes.titleIcon} src={logo} alt="Logo" />
                    <div className={classes.titleText}>
                        {strings.not_logged_in_title}
                    </div>
                    <div className={classes.subTitleText}>
                        {strings.not_logged_in_sub_title}
                    </div>
                </div>
                <div className={classes.rightContainer}>
                    {this.renderRightSide()}
                </div>
                <div
                    className={classes.supportContainer}
                    onClick={() =>
                        shell.openExternal(
                            device.model === MODELS.TERRANOXT
                                ? TERRANO_SUPPORT_URL
                                : SUPPORT_URL
                        )
                    }
                >
                    <img
                        className={classes.supportIcon}
                        src={supportIcon}
                        alt="Support"
                    />
                    {strings.support}
                </div>
            </div>
        );
    }
}

export default withStyles(styles)(WelcomeScreen);
